import React from 'react';

const DOCUMENT_TYPES = [
  { id: '1', name: 'Passport' },
  { id: '2', name: 'Photo' },
  { id: '3', name: 'Residency' },
  { id: '4', name: 'Other' },
  { id: '5', name: 'ID Card' },
];

interface TypeSelectorProps {
  selectedType: string;
  onChange: (typeId: string) => void;
}

export function TypeSelector({ selectedType, onChange }: TypeSelectorProps) {
  React.useEffect(() => {
    const handleKeyPress = (e: KeyboardEvent) => {
      const key = Number(e.key);
      if (key >= 1 && key <= DOCUMENT_TYPES.length) {
        onChange(DOCUMENT_TYPES[key - 1].id);
      }
    };

    window.addEventListener('keypress', handleKeyPress);
    return () => window.removeEventListener('keypress', handleKeyPress);
  }, [onChange]);

  return (
    <div className="space-y-4">
      <div className="text-sm text-gray-500">
        Press 1-5 to quickly select document types
      </div>
      <div className="grid grid-cols-1 gap-3">
        {DOCUMENT_TYPES.map((type, index) => (
          <label
            key={type.id}
            className="flex items-center space-x-3 p-3 rounded-lg border cursor-pointer hover:bg-gray-50 transition-colors"
          >
            <input
              type="radio"
              name="document-type"
              value={type.id}
              checked={selectedType === type.id}
              onChange={() => onChange(type.id)}
              className="h-4 w-4 text-blue-600 focus:ring-blue-500"
            />
            <span className="text-gray-900">
              {index + 1}. {type.name}
            </span>
          </label>
        ))}
      </div>
    </div>
  );
}